<?php

namespace App\Http\Controllers\Merchant;

use App\Http\Controllers\Controller;
use App\Http\Requests\SalesPerson\StoreSalesPersonRequest;
use App\Http\Requests\SalesPerson\UpdateSalesPersonRequest;
use App\Models\SalesPerson;
use App\Models\User;
use Illuminate\Http\Request;
use DataTables;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Hash;

class SalesPersonController extends Controller
{
    public function __construct()
    {
        $this->defaultPassword = "Suhu@123";
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if(\Auth::user()->can('list sales person'))
        {
            if ($request->ajax()) {
        
                $merchant = User::getMerchant(\Auth::user()->id);   
                $merchant_id = $merchant->id;
                $results = SalesPerson::where('merchant_id',$merchant_id)
                                ->get();

                return DataTables::of($results)
                ->addIndexColumn()
                ->addColumn("first_name", function ($data) {
                    if (empty($data->first_name)) {
                        return "N/A";
                    }
                    return $data->first_name;
                })
               
                ->addColumn("email", function ($data) {
                    if (empty($data->email)) {
                        return "N/A";
                    }
                    return $data->email;
                })
                ->addColumn("phone", function ($data) {
                    if (empty($data->phone)) {
                        return "N/A";
                    }
                    return $data->phone;
                })
                ->addColumn("action", function ($data) {
                    
                    $btn =
                        '<a class="btn btn-info btn-xs" href="' .
                        route("merchant.sales-person.edit", $data->id) .
                        '" data-toggle="tooltip" data-placement="top" title="Edit Sales Person"><i class="fa fa-edit"></i> Edit</a>&nbsp;';
                    $btn .=
                        '<a class="btn btn-danger btn-xs deletesalesperson" href="javascript:;" data-customer="' .
                        $data->id .
                        '" data-toggle="tooltip" data-placement="top" title="Delete Sales Person"><i class="fa fa-trash"></i> Delete</a>';
                    return $btn;
                })
                ->rawColumns(["action"])
                ->make(true);
            }

            $data["title"] = "Manage Sales Person";

            return view("pages.merchant.sales-person.list", $data);  
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if(\Auth::user()->can('create sales person'))
        {     
            $data["password"] = $this->defaultPassword;       
            return view('pages.merchant.sales-person.create',$data);
        }    
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreSalesPersonRequest $request)
    {
        if(\Auth::user()->can('create sales person'))
        {
            $is_exist = SalesPerson::where('email', $request->email)->exists();
        
            if($is_exist)
            {
                return redirect()
                ->route("merchant.sales-person.index")
                ->with("failed", "Account already exist");         
            }else{
                
                try
                {
                    
                    $merchant = User::getMerchant(\Auth::user()->id);   
                    $merchant_id = $merchant->id;
                    $salesPersonCount = SalesPerson::where('merchant_id',$merchant_id)->count();
                    
                    if($salesPersonCount>3)
                    {
                        return redirect()
                        ->route("merchant.sales-person.index")
                        ->with("failed", "You allowed to create maximum 3 sales persons only. please contact admin");
                    }
                    else
                    {
                        try
                        {                            
                            $user_id = User::createUser($request,'SALES_PERSON');
                            if($user_id)
                            {
                                $password = Hash::make($request->password);
                                $sales_person = SalesPerson::create([
                                    "user_id" => $user_id,
                                    "merchant_id" => $merchant_id,            
                                    "first_name" => trim($request->first_name),
                                    "last_name" => trim($request->last_name),
                                    "user_name" => trim($request->first_name).' '.trim($request->last_name),
                                    "email" => trim($request->email),                       
                                    "phone" => trim($request->phone),                        
                                    "created_by" => \Auth::user()->id,
                                    "password" => trim($password),
                                ]);
                            }
                        }
                        catch(QueryException $ex) 
                        {
                            return redirect()
                            ->route("sales-person.index")
                            ->with("failed","User Insertion Error");                 
                        } 
                    }
                    
                }
                catch(QueryException $ex)
                {
                    return redirect()
                    ->route("merchant.sales-person.index")
                    ->with("failed", "Insert Error");                 
                }
                    
            }
            
            return redirect()
            ->route("merchant.sales-person.index")
            ->with("success", "Sales Person created successfully!");
        }    
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if(\Auth::user()->can('edit sales person'))
        {
            $salesPersonInfo = SalesPerson::find($id);
            $data["salesPersonInfo"] = $salesPersonInfo;            
            return view('pages.merchant.sales-person.edit',$data);
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }  
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateSalesPersonRequest $request, $id)
    {
        if(\Auth::user()->can('edit sales person'))
        {
            try
            {
                $merchant = User::getMerchant(\Auth::user()->id);
                $merchant_id = $merchant->id;
                $salesPersonCount = SalesPerson::where('merchant_id',$merchant_id)->count();                    
                if($salesPersonCount>3)
                {
                    return redirect()
                    ->route("merchant.sales-person.index")
                    ->with("failed", "You allowed to create maximum 3 sales persons only. please contact admin..");
                }
                else
                {
                    $salesPerson = SalesPerson::find($id);
                    $salesPerson->merchant_id = $merchant_id;
                    $salesPerson->first_name = $request->first_name;
                    $salesPerson->last_name = $request->last_name;
                    $salesPerson->user_name = $request->first_name.' '.$request->last_name;
                    $salesPerson->email = $request->email;
                    $salesPerson->phone = $request->phone;
                    $salesPerson->created_by = \Auth::user()->id;
                    $salesPerson->save();
                }
            }
            catch(QueryException $ex) 
            {
                return redirect()
                ->route("merchant.sales-person.index")
                ->with("failed", "Insert Error");                 
            }
            return redirect()
                ->route("merchant.sales-person.index")
                ->with("success", "Sales Person updated successfully!");
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        if(\Auth::user()->can('delete sales person'))
        {
            $salesPerson = SalesPerson::find($request->id);
            $salesPerson->delete();
            Session::flash("success", "Sales Person information deleted successfully.");
            return 1;
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}
